/**
 * A wrapper around the Browser History Manger.  It converts a state object to a string and shortens it using WR.TextShortener.
 * 
 * Example:
 *
 * In initialization:
 *
 * this.history = new WR.History( 'myModule', this, {
 *  generateState: this.generateState,
 *  loadInitialState: this.loadInitialState,
 *  onStateChanged: this.onStateChanged
 *  } );
 *
 * When you want to update the browser history:
 *
 * this.history.update();
 * 
 *
 *
 * generateState( scope ) should return an object representing the state of the module.
 * 
 * loadInitialState( state, scope ) should apply the state of the module.  This will be be called after WR.History.initialize() is called.
 *      The state to apply will be either the bookmarked state or null if there is no bookmarked state.
 *      
 * onStateChanged( newState, scope ) will be called every time history.update() is called and the current state is different than the last
 *      time history.update() was called.
 *
 * The 'scope' argument is the object you passed in as the second argument to the constructor.
 *
 */

// <?import /JavaScript/Notification/Notification.js?>
// <?import /JavaScript/yui-import/Event.js?>
// <?import /JavaScript/json2.js?>
// <?import /JavaScript/Util/Object.js?>
// <?import /JavaScript/Util/History.js?>
// <?import /JavaScript/yui-import/BrowserHistoryManager.js?>
// <?import /JavaScript/Util/TextShortener.js?>
// <?import /JavaScript/Util/JSONRequest.js?>
// <?import /JavaScript/Util/Set.js?>


WR.History = function( moduleName, methodScope, opts )
{
    this.scope = methodScope;
    this.module = moduleName;
    
    this.ignoreUpdates = false;
    this._inNavigation = false;
    
    this.currentStateObject = {};
    
    if ( opts )
    {
        this.generateState = opts.generateState? opts.generateState : function( scope ) { return {}; };
        this.loadInitialState = opts.loadInitialState? opts.loadInitialState : function( state, scope ) { };
        this.onStateChanged = opts.onStateChanged? opts.onStateChanged : function( newState, oldState, scope ) { };
    }
    
    this._init();
};

WR.History.prototype = {
    
    runWhileIgnoringUpdates : function( fn )
    {
        var wasIgnoringUpdates = this.ignoreUpdates;
        this.ignoreUpdates = true;
        fn();
        this.ignoreUpdates = wasIgnoringUpdates;
    },
    
    update : function()
    {
        if ( this.ignoreUpdates )
            return;
        
        var newStateObject = this.generateState( this.scope );
        if ( WR.Object.equal( newStateObject, this.currentStateObject ) )
            return;
        
        var newState = WR.History.generateStateURL( newStateObject );
        
        if ( WR.History.isLoadComplete )
            this._navigate( newState );
        this.onStateChanged( newStateObject, this.currentStateObject, this.scope );
        WR.History.onChange.fire();
        this.currentStateObject = newStateObject;
    
    },
    
    _navigate : function( newState )
    {
        this._inNavigation = true;
        YAHOO.util.History.navigate( this.module, newState );
    },
    
    _onModuleStateChange : function( state )
    {
        if ( !this._inNavigation && state != '' )
            location.reload( true );
        
        this._inNavigation = false;
    },
    
    _init : function()
    {
        var bookmarkedState = YAHOO.util.History.getBookmarkedState( this.module );
        
        var me = this;
        if ( bookmarkedState )
        {
            YAHOO.util.History.register( this.module, bookmarkedState,
                function( state )
                {
                    me._onModuleStateChange( state );
                }
            );
                           
            YAHOO.util.History.onReady( function() {
                
                if ( bookmarkedState.match( /[a-f0-9]{40}/i ) )
                {
                    WR.History._openLoadConnections++;
                    WR.TextShortener.UnshortenText( bookmarkedState, {
                        success: function( fullText ) {
                            me.currentStateObject = JSON.parse( fullText );
                            me.loadInitialState( me.currentStateObject, me.scope );
                            WR.History._openLoadConnections--;
                            WR.History.checkIfLoadComplete();
                        },
                        failure: function( errorText )
                        {
                            WR.Notify.error( WR.format( '<?jformat Invalid state for module {module}: {errorText}?>', me.module, errorText ) );
                            WR.History._openLoadConnections--;
                            WR.History.checkIfLoadComplete();
                        }
                    } );
                }
                else
                {
                    me.currentStateObject = JSON.parse( bookmarkedState );
                    me.loadInitialState( me.currentStateObject, me.scope );
                }
            } );
        }
        else
        {
            this.currentStateObject = this.generateState( this.scope );
            
            //Register empty string as the default state so we don't have to ask server for shortened URL
            YAHOO.util.History.register( this.module, '',
                function( state )
                {
                    me._onModuleStateChange( state );
                }
            );
            
            YAHOO.util.History.onReady( function() {
                me.loadInitialState( null, me.scope );
            } );
        }
    }
    
};

WR.History.initialize = function() {
    try
    {
        YAHOO.util.History.initialize( 'yui-history-field', 'yui-history-iframe' );
        
        if ( YAHOO.env.ua.ie == 7 && WR.URL.ParseParams( top.location.hash )['IsRedirect'] == 1 )
        {
            //If we were redirected to this page, setting top.location.hash will cause the page to reload instead of changing the hash.  See bug 35444.
            var oldHash = top.location.hash;
            top.location.hash = 'history_detection';
            top.location.hash = oldHash;
        }
    }
    catch ( e ) // If browser isn't supported, such as Opera =(
    {
        //do nothing?
    }
};

WR.History.loadTimeSafeJSONRequest = function ( url, callbacks, params /* optional */, scope /* optional */ )
{
    if ( WR.History.isLoadComplete )
    {
        WR.jsonRequest( url, callbacks, params, scope );
        return;
    }
    
    WR.History._openLoadConnections++;
    
    var newCallbacks = {
        success: function( result, scope )
        {
            callbacks.success( result, scope );
            WR.History._openLoadConnections--;
            WR.History.checkIfLoadComplete();
        },
        
        failure: function( result, scope )
        {
            callbacks.failure( result, scope );
            WR.History._openLoadConnections--;
            WR.History.checkIfLoadComplete();
        }
    };
    
    WR.jsonRequest( url, newCallbacks, params, scope );
}

WR.History.checkIfLoadComplete = function()
{
    if ( WR.History._openLoadConnections == 0 && WR.History.loadEventFired )
    {
        WR.History.isLoadComplete = true;
        WR.History.onLoadComplete.fire();
    }
}

WR.History.generateStateURL = function( state, callbacks )
{
    return WR.TextShortener.ShortenText( JSON.stringify( state ), callbacks );
}

WR.History.pageStatesAreEqual = function( urlA, urlB )
{
    function hashParams( url )
    {
        return WR.URL.ParseParams( WR.URL.getHash( decodeURIComponent( url ) ) ); 
    }
    
    //////
    
    if ( WR.URL.getQuery( urlA ) != WR.URL.getQuery( urlB ) )
        return false;
    
    var paramsA = hashParams( urlA );
    var paramsB = hashParams( urlB );
    
    var module;
    for ( module in paramsA )
    {
        if ( paramsB[module] == null )
            paramsB[module] = [''];
    }
    for ( module in paramsB )
    {
        if ( paramsA[module] == null )
            paramsA[module] = [''];
    }
    
    for ( module in paramsA )
    {
        if ( paramsA[module][0] != paramsB[module][0] )
            return false;
    }
    
    return true;
}

WR.History.onChange = new YAHOO.util.CustomEvent( 'onChange' );

//Called from within YUI's history.js
WR.History.onLoadComplete = new YAHOO.util.CustomEvent( 'onLoadComplete' );
WR.History.isLoadComplete = false;
WR.History._openLoadConnections = 0;
WR.History.loadEventFired = false;